%results = {};
use Net::Telnet;

################################################################################
###    Public Functions                                                      ###
################################################################################

################################################################################
# INPUT:
# getinfo( DeviceName, TermservName, TermservPort, Command );
# getinfo( 'gd60_m84_01', 't12', '2001', 'show hardware');
#   note: DeviceName and TermservName need to resolve with DNS. The IP
#         address may also be used. If you use the IP address for DeviceName,
#         every subsequent call with the associated physical device should pass
#         in the IP address as well.
# getinfo( '14.0.0.100', '172.20.61.12', '2001', 'show hardware');
#
# OUTPUT: 
#   Returns a scalar storing the result of running the specified command on a
#   particular device. The library will only run the command once and save
#   the output for future referance. Returns "-1" if there were errors
#   connecting to the device.
################################################################################

sub getinfo {
   my ($dev, $terms, $terms_p, $cmd) = @_;  # name, termserv name, port, command

   if (check_errors($dev)) {                # stop if previous connection errors
      return -1;
   }
   elsif ( prev_run($dev,$cmd) ) {          # lookup if command already run
      return lookup($dev,$cmd);
   }
   else {                                   # run the command
      if ( !connected($dev) ) {             # if not connected: connect
         if ( !enet_connect($dev) && $opt_x ) { # if ethernet fails, try console
            console_connect($dev, $terms, $terms_p);
         }
      }
      if (check_errors($dev)) {             # stop if connection errors
         return -1;
      }
      run_cmd($dev,$cmd);
      return lookup($dev,$cmd);
   }
}

################################################################################
# INPUT:
# connection_status( Device Name );
# connection_status( 'gd60_m84_01' );  *or*  connection_status('14.0.0.100');
#
# OUTPUT:
#   Returns the scalar value corresponding to the current connection status of
#   the stated device. Possible values are: Ethernet, Console,
#   ethernet-timed-out, nopasswd, console in use, console timed-out, monitor, 0,
#   and undefined:(returned error). Ethernet and Console both imply that the
#   connection was/is established correctly. Any other value implies that a
#   connection was not established.
################################################################################
sub connection_status {
   my ($dev) = @_;
   if (check_errors($dev)) {             # An error occured in connecting
      return $con_errors{"$dev"};           # nopasswd, console in use, etc.
   }
   elsif (connected($dev)) {                   # A connection is open
      return $connected_via{"$dev"};        # Ethernet or Console
   }
   else {
      return 'unknown';
   }
}
      
################################################################################
# INPUT:  array of keys to %connections *or* none
# OUTPUT: none
# Closes requested *or* all open connections.
# BETA Notes: does not update the @connected_to array
################################################################################
sub disconnect {
   if (@_) {
      @dump_connections_to = @_;
   }
   else {
      @dump_connections_to = @connected_to;
   }
   foreach $dev (@dump_connections_to) {
      if ($connections{"$dev"}) {
         $connections{"$dev"}->print("exit");
         $connections{"$dev"}->print("exit");
         $connections{"$dev"}->close;
      }
   }
}

################################################################################
###    Private Functions                                                     ###
################################################################################

sub prev_run {                              # Has cmd been run on dev?
   my ($dev, $cmd) = @_;
   if ( exists $results{"$dev"}->{"$cmd"} ) {
      return 1;                             # cmd has been previously run on dev
   } 
   else {
      return 0;                             # cmd has not been run on dev
   }
}

sub lookup {                                # return results from cmd on dev
   my ($dev, $cmd) = @_;
   return $results{"$dev"}->{"$cmd"};
}

sub connected {                             # Is there a connection to dev?
   my ($dev) = @_;
   if ($connections{"$dev"}) {              # If the telnet object exists
      return 1;
   }
   else {
      return 0;
   }
}

sub check_errors {                          # Any errors while connecting?
   my ($dev) = @_;
   if ( $con_errors{"$dev"} ) {
      return 1;
   }
   else {
      return 0;
   }
}

################################################################################
# INPUT: DeviceName (or IP)
# OUTPUT: 1 (success) or 0 (failed)
# Attempts to establish an telnet connection with the specified device via the
# device's ethernet port.
################################################################################
sub enet_connect {
   my ($dev) = @_;
   $connections{"$dev"} = new Net::Telnet;
   $connections{"$dev"}->prompt('/[#>]$/');
   $connections{"$dev"}->errmode("return");
   $connections{"$dev"}->timeout("3");
   $connections{"$dev"}->open($dev);
   $connections{"$dev"}->waitfor('/(Password:)|[#>].*$/');

   if ($connections{"$dev"}->errmsg) {
      my $Out = $connections{"$dev"}->errmsg;
      if ($Out =~ /connection timed-out/) {
         $con_errors{"$dev"} = "ethernet-timed-out";
      }
      elsif ($Out =~ /pattern match read eof/) {
         $con_errors{"$dev"} = "nopasswd";
      }
      else {
         print "Ethernet undefined error: $Out\n";
         $con_errors{"$dev"} = "undefined:$Out";
      }
      $connections{"$dev"}->close;
      return 0;
   }
   else {
      $con_errors{"$dev"} = 0;
      push(@connected_to,($dev));
      $connected_via{"$dev"} = "Ethernet";
      $connections{"$dev"}->print("lab");
      $connections{"$dev"}->getlines;
      $connections{"$dev"}->cmd("en");
      $connections{"$dev"}->cmd("lab");
      $connections{"$dev"}->cmd("term length 0");
      return 1;
   }
}

################################################################################
# INPUT: console_connect( DeviceName-or-IP, TermservName, TermservPort );
#
# OUTPUT: 1 (success) or 0 (failed)
# Attempts to establish an telnet connection with the specified device through
# the console port via the terminal server.
################################################################################
sub console_connect {
   #print "Console connection...\n";
   my ($dev, $termserv, $port_num) = @_;

   $connections{"$dev"} = Net::Telnet->new(
                                 Errmode    => 'return',
                                 Host       => $termserv,
                                 Port       => $port_num,
                                 Timeout    => 4,
                                 Prompt     => '/(Password:)|[#>]$/',
                                 );
   if (!$connections{"$dev"}) {
      #print "console in use\n";
      if ($con_errors{"$dev"} ne 
                  "undefined:problem creating socket: Too many open files" ) {
         $con_errors{"$dev"} = "console in use";
      }
      return 0;
   }
   elsif ($connections{"$dev"}->errmsg) {
      my $Out = $connections{"$dev"}->errmsg;

      if ($Out =~ /connection timed-out/) {
         $con_errors{"$dev"} = "console timed-out";
      }
      else {
         print "Console undefined error: $Out\n";
         $con_errors{"$dev"} = "undefined:\$Out";
      }

      return 0;
   }
   else {
      $con_errors{"$dev"} = 0;
      foreach $cmd_line ($connections{"$dev"}->cmd("Hello")) {
         if ($cmd_line =~ /monitor/) {
            $con_errors{"$dev"} = "monitor";
            $connections{"$dev"}->close();
            return 0;
         }
         #print "Monitor Check: $cmd_line\n";
      }
      push(@connected_to,($dev));
      $connected_via{"$dev"} = "Console";
      $connections{"$dev"}->print("");
      $connections{"$dev"}->cmd("en");
      $connections{"$dev"}->cmd("lab");
      $connections{"$dev"}->cmd("term length 0");

      return 1;
   }
}

sub run_cmd {                               # Run cmd on dev and store output
   my ($dev, $cmd) = @_;
   $results{"$dev"}->{"$cmd"} = "";

   foreach $cmd_line ($connections{"$dev"}->cmd("$cmd")) {
      $results{$dev}->{$cmd} = "$results{$dev}->{$cmd}$cmd_line";
   }
}

1;
